// SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB
/* Copyright (c) 2019 - 2020 Intel Corporation */
#include "osdep.h"
#include "type.h"
#include "icrdma_hw.h"
#include "main.h"

static u32 icrdma_regs[IRDMA_MAX_REGS] = {
	PFPE_CQPTAIL,
	PFPE_CQPDB,
	PFPE_CCQPSTATUS,
	PFPE_CCQPHIGH,
	PFPE_CCQPLOW,
	PFPE_CQARM,
	PFPE_CQACK,
	PFPE_AEQALLOC,
	PFPE_CQPERRCODES,
	PFPE_WQEALLOC,
	GLINT_DYN_CTL(0),
	ICRDMA_DB_ADDR_OFFSET,

	GLPCI_LBARCTRL,
	GLPE_CPUSTATUS0,
	GLPE_CPUSTATUS1,
	GLPE_CPUSTATUS2,
	PFINT_AEQCTL,
	GLINT_CEQCTL(0),
	VSIQF_PE_CTL1(0),
	PFHMC_PDINV,
	GLHMC_VFPDINV(0),
	GLPE_CRITERR,
	GLINT_RATE(0),
};

static u64 icrdma_masks[IRDMA_MAX_MASKS] = {
	ICRDMA_CCQPSTATUS_CCQP_DONE_M,
	ICRDMA_CCQPSTATUS_CCQP_ERR_M,
	ICRDMA_CQPSQ_STAG_PDID_M,
	ICRDMA_CQPSQ_CQ_CEQID_M,
	ICRDMA_CQPSQ_CQ_CQID_M,
	ICRDMA_COMMIT_FPM_CQCNT_M,
};

static u64 icrdma_shifts[IRDMA_MAX_SHIFTS] = {
	ICRDMA_CCQPSTATUS_CCQP_DONE_S,
	ICRDMA_CCQPSTATUS_CCQP_ERR_S,
	ICRDMA_CQPSQ_STAG_PDID_S,
	ICRDMA_CQPSQ_CQ_CEQID_S,
	ICRDMA_CQPSQ_CQ_CQID_S,
	ICRDMA_COMMIT_FPM_CQCNT_S,
};

void icrdma_init_hw(struct irdma_sc_dev *dev)
{
	int i;
	u8 __iomem *hw_addr;

	for (i = 0; i < IRDMA_MAX_REGS; ++i) {
		hw_addr = dev->hw->hw_addr;

		if (i == IRDMA_DB_ADDR_OFFSET)
			hw_addr = NULL;

		dev->hw_regs[i] = (u32 __iomem *)(hw_addr + icrdma_regs[i]);
	}
	dev->hw_attrs.max_hw_vf_fpm_id = IRDMA_MAX_VF_FPM_ID;
	dev->hw_attrs.first_hw_vf_fpm_id = IRDMA_FIRST_VF_FPM_ID;

	for (i = 0; i < IRDMA_MAX_SHIFTS; ++i)
		dev->hw_shifts[i] = icrdma_shifts[i];

	for (i = 0; i < IRDMA_MAX_MASKS; ++i)
		dev->hw_masks[i] = icrdma_masks[i];

	dev->wqe_alloc_db = dev->hw_regs[IRDMA_WQEALLOC];
	dev->cq_arm_db = dev->hw_regs[IRDMA_CQARM];
	dev->aeq_alloc_db = dev->hw_regs[IRDMA_AEQALLOC];
	dev->cqp_db = dev->hw_regs[IRDMA_CQPDB];
	dev->cq_ack_db = dev->hw_regs[IRDMA_CQACK];
	dev->hw_attrs.max_stat_inst = ICRDMA_MAX_STATS_COUNT;

	dev->hw_attrs.uk_attrs.max_hw_sq_chunk = IRDMA_MAX_QUANTA_PER_WR;
}

void irdma_init_config_check(struct irdma_config_check *cc, u8 traffic_class, u16 qs_handle)
{
	cc->config_ok = false;
	cc->traffic_class = traffic_class;
	cc->qs_handle = qs_handle;
	cc->lfc_set = 0;
	cc->pfc_set = 0;
}

static bool irdma_is_lfc_set(struct irdma_config_check *cc, struct irdma_sc_vsi *vsi)
{
	u32 lfc = 1;
	u8 fn_id = vsi->dev->hmc_fn_id;

	lfc &= (rd32(vsi->dev->hw,
		     PRTMAC_HSEC_CTL_RX_PAUSE_ENABLE_0 + 4 * fn_id) >> 8);
	lfc &= (rd32(vsi->dev->hw,
		     PRTMAC_HSEC_CTL_TX_PAUSE_ENABLE_0 + 4 * fn_id) >> 8);
	lfc &= rd32(vsi->dev->hw,
		    PRTMAC_HSEC_CTL_RX_ENABLE_GPP_0 + 4 * vsi->dev->hmc_fn_id);

	if (lfc)
		return true;
	return false;
}

static bool irdma_check_tc_has_pfc(struct irdma_sc_vsi *vsi, u64 reg_offset, u16 traffic_class)
{
	u32 value, pfc = 0;
	u32 i;

	value = rd32(vsi->dev->hw, reg_offset);
	for (i = 0; i < 4; i++)
		pfc |= (value >> (8 * i + traffic_class)) & 0x1;

	if (pfc)
		return true;
	return false;
}

static bool irdma_is_pfc_set(struct irdma_config_check *cc, struct irdma_sc_vsi *vsi)
{
	u32 pause;
	u8 fn_id = vsi->dev->hmc_fn_id;

	pause = (rd32(vsi->dev->hw,
		      PRTMAC_HSEC_CTL_RX_PAUSE_ENABLE_0 + 4 * fn_id) >>
		      cc->traffic_class) & BIT(0);
	pause &= (rd32(vsi->dev->hw,
		       PRTMAC_HSEC_CTL_TX_PAUSE_ENABLE_0 + 4 * fn_id) >>
		       cc->traffic_class) & BIT(0);

	return irdma_check_tc_has_pfc(vsi, GLDCB_TC2PFC, cc->traffic_class) &&
	       pause;
}

bool irdma_is_config_ok(struct irdma_config_check *cc, struct irdma_sc_vsi *vsi)
{
	cc->lfc_set = irdma_is_lfc_set(cc, vsi);
	cc->pfc_set = irdma_is_pfc_set(cc, vsi);

	cc->config_ok = cc->lfc_set || cc->pfc_set;

	return cc->config_ok;
}

#define IRDMA_RCV_WND_NO_FC	0x1FFFC
#define IRDMA_RCV_WND_FC	0x3FFFC

#define IRDMA_CWND_NO_FC	0x20
#define IRDMA_CWND_FC		0x400

#define IRDMA_RTOMIN_NO_FC	0x5
#define IRDMA_RTOMIN_FC		0x32

#define IRDMA_ACKCREDS_NO_FC	0x02
#define IRDMA_ACKCREDS_FC	0x1E

void irdma_check_flow_ctrl(struct irdma_sc_vsi *vsi, u8 user_prio, u8 traffic_class)
{
	struct irdma_config_check *cfg_chk = &vsi->cfg_check[user_prio];
	struct irdma_device *iwdev = vsi->back_vsi;

	if (!irdma_is_config_ok(cfg_chk, vsi)) {
		if (!iwdev->override_rcv_wnd)
			iwdev->rcv_wnd = IRDMA_RCV_WND_NO_FC;
		if (!iwdev->override_cwnd)
			iwdev->roce_cwnd = IRDMA_CWND_NO_FC;
		if (!iwdev->override_rtomin)
			iwdev->roce_rtomin = IRDMA_RTOMIN_NO_FC;
		if (!iwdev->override_ackcreds)
			iwdev->roce_ackcreds = IRDMA_ACKCREDS_NO_FC;
#define IRDMA_READ_FENCE_RATE_NO_FC	4
		if (iwdev->roce_mode && !iwdev->override_rd_fence_rate)
			iwdev->rd_fence_rate = IRDMA_READ_FENCE_RATE_NO_FC;
		if (vsi->tc_print_warning[traffic_class]) {
			pr_info("INFO: Flow control is disabled for this traffic class (%d) on this vsi.\n",
				traffic_class);
			vsi->tc_print_warning[traffic_class] = false;
		}
	} else {
		if (!iwdev->override_rcv_wnd)
			iwdev->rcv_wnd = IRDMA_RCV_WND_FC;
		if (!iwdev->override_cwnd)
			iwdev->roce_cwnd = IRDMA_CWND_FC;
		if (!iwdev->override_rtomin)
			iwdev->roce_rtomin = IRDMA_RTOMIN_FC;
		if (!iwdev->override_ackcreds)
			iwdev->roce_ackcreds = IRDMA_ACKCREDS_FC;
#define IRDMA_READ_FENCE_RATE_FC	0
		if (!iwdev->override_rd_fence_rate)
			iwdev->rd_fence_rate = IRDMA_READ_FENCE_RATE_FC;
		if (vsi->tc_print_warning[traffic_class]) {
			pr_info("INFO: Flow control is enabled for this traffic class (%d) on this vsi.\n",
				traffic_class);
			vsi->tc_print_warning[traffic_class] = false;
		}
	}
}

void irdma_check_fc_for_tc_update(struct irdma_sc_vsi *vsi,
				  struct irdma_l2params *l2params)
{
	u8 i;

	for (i = 0; i < IRDMA_MAX_TRAFFIC_CLASS; i++)
		vsi->tc_print_warning[i] = true;

	for (i = 0; i < IRDMA_MAX_USER_PRIORITY; i++) {
		struct irdma_config_check *cfg_chk = &vsi->cfg_check[i];
		u8 tc = l2params->up2tc[i];

		cfg_chk->traffic_class = tc;
		cfg_chk->qs_handle = vsi->qos[i].qs_handle;
		irdma_check_flow_ctrl(vsi, i, tc);
	}
}

void irdma_check_fc_for_qp(struct irdma_sc_vsi *vsi, struct irdma_sc_qp *sc_qp)
{
	u8 i;

	for (i = 0; i < IRDMA_MAX_USER_PRIORITY; i++) {
		struct irdma_config_check *cfg_chk = &vsi->cfg_check[i];

		irdma_init_config_check(cfg_chk,
					vsi->qos[i].traffic_class,
					vsi->qos[i].qs_handle);
		if (sc_qp->qs_handle == cfg_chk->qs_handle)
			irdma_check_flow_ctrl(vsi, i, cfg_chk->traffic_class);
	}
}

