/* SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB */
/* Copyright (c) 2019 - 2021 Intel Corporation */
#ifndef IRDMA_KCOMPAT_H
#define IRDMA_KCOMPAT_H

#ifndef LINUX_VERSION_CODE
#include <linux/version.h>
#else
#define KERNEL_VERSION(a, b, c) (((a) << 16) + ((b) << 8) + (c))
#endif
#include <linux/init.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/list.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/in.h>
#include <linux/ip.h>
#include <linux/udp.h>
#include <linux/mii.h>
#include <linux/vmalloc.h>
#include <linux/irq.h>
#include <linux/hugetlb.h>
#include <asm/io.h>
#include <linux/ethtool.h>
#include <linux/if_vlan.h>
#include <net/ipv6.h>
#include <net/ip6_route.h>
#include <net/route.h>
#include <rdma/ib_cache.h>
#include <rdma/ib_umem.h>
#if IS_ENABLED(CONFIG_NET_DEVLINK)
#include <net/devlink.h>
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 14, 0)
#include <rdma/uverbs_ioctl.h>
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 4, 0)
#include <linux/kconfig.h>
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 19, 0)
#include <net/secure_seq.h>
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 4, 0)
#include <asm-generic/io-64-nonatomic-lo-hi.h>
#else
#include <linux/io-64-nonatomic-lo-hi.h>
#endif

#include "distro_ver.h"

#ifdef __OFED_BUILD__
#include "ofed_kcompat.h"
#elif defined(RHEL_RELEASE_CODE)
#include "rhel_kcompat.h"
#elif defined(CONFIG_SUSE_KERNEL)
#include "suse_kcompat.h"
#elif defined(UTS_UBUNTU_RELEASE_ABI) && (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 4, 0))
#include "ubuntu_kcompat.h"
#else
#include "linux_kcompat.h"
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION(5, 10, 0))
#define TASKLET_DATA_TYPE	unsigned long
#define TASKLET_FUNC_TYPE	void (*)(TASKLET_DATA_TYPE)

#define tasklet_setup(tasklet, callback)				\
	tasklet_init((tasklet), (TASKLET_FUNC_TYPE)(callback),		\
		      (TASKLET_DATA_TYPE)(tasklet))

#define from_tasklet(var, callback_tasklet, tasklet_fieldname) \
	container_of(callback_tasklet, typeof(*var), tasklet_fieldname)
#endif /* (LINUX_VERSION_CODE < KERNEL_VERSION(5, 10, 0)) */

/* Mapping IRDMA driver ID to I40IW till we are in k.org */
#define RDMA_DRIVER_IRDMA RDMA_DRIVER_I40IW

#if (LINUX_VERSION_CODE < KERNEL_VERSION(4, 14, 0))
#define TIMER_DATA_TYPE		unsigned long
#define TIMER_FUNC_TYPE		void (*)(TIMER_DATA_TYPE)

#define timer_setup(timer, callback, flags)				\
	__setup_timer((timer), (TIMER_FUNC_TYPE)(callback),		\
		      (TIMER_DATA_TYPE)(timer), (flags))

#define from_timer(var, callback_timer, timer_fieldname) \
	container_of(callback_timer, typeof(*var), timer_fieldname)
#endif /* (LINUX_VERSION_CODE < KERNEL_VERSION(4, 14, 0)) */

#ifndef __OFED_BUILD__
#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 0, 0)
#define dma_alloc_coherent dma_zalloc_coherent
#endif
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 0, 0)
#define IB_GET_NETDEV_OP_NOT_DEPRECATED
#endif

/*************************QUERY PKEY*********************************************/
/* https://lore.kernel.org/linux-rdma/9DD61F30A802C4429A01CA4200E302A7010659C72C@fmsmsx124.amr.corp.intel.com/
 * This series removes query pkey callback from iWARP providers as it really
 * is not required as per the protocol. Also IB core is updated to not expose
 * pkey related sysfs attributes for iw_devices. Prior to 5.9, query pkey is mandatory
 * for iWARP providers.
 */
#ifdef IB_IW_PKEY
static inline int irdma_iw_query_pkey(struct ib_device *ibdev, u8 port, u16 index,
				      u16 *pkey)
{
	*pkey = 0;
	return 0;
}
#endif
/*******************************************************************************/

struct dst_entry *irdma_get_fl6_dst(struct sockaddr_in6 *, struct sockaddr_in6 *);
struct neighbour *irdma_get_neigh_ipv6(struct dst_entry *, struct sockaddr_in6 *);
struct neighbour *irdma_get_neigh_ipv4(struct rtable *, __be32 *);
void kc_set_loc_seq_num_mss(struct irdma_cm_node *cm_node);

struct irdma_mr;
struct irdma_cq;
struct irdma_cq_buf;
u32 irdma_create_stag(struct irdma_device *iwdev);
void irdma_free_stag(struct irdma_device *iwdev, u32 stag);
int irdma_hw_alloc_mw(struct irdma_device *iwdev, struct irdma_mr *iwmr);
void irdma_cq_free_rsrc(struct irdma_pci_f *rf, struct irdma_cq *iwcq);
int irdma_process_resize_list(struct irdma_cq *iwcq, struct irdma_device *iwdev, struct irdma_cq_buf *lcqe_buf);

/*************************DEVLINK*********************************************/
#if IS_ENABLED(CONFIG_NET_DEVLINK)
#ifdef __DEVLINK_PARAM_MAX_STRING_VALUE
#define DEVLINK_SUPPORTED
#endif
#endif
/*********************************************************/

#ifdef DEVLINK_SUPPORTED
enum irdma_dl_param_id {
	IRDMA_DEVLINK_PARAM_ID_BASE = DEVLINK_PARAM_GENERIC_ID_MAX,
	IRDMA_DEVLINK_PARAM_ID_LIMITS_SELECTOR,
	IRDMA_DEVLINK_PARAM_ID_UPLOAD_CONTEXT,
#ifndef DEVLINK_PARAM_GENERIC_ENABLE_ROCE_TYPE
	IRDMA_DEVLINK_PARAM_ID_ROCE_ENABLE,
#endif
	/* DCQCN congestion control */
	IRDMA_DEVLINK_PARAM_ID_DCQCN_ENABLE,
	IRDMA_DEVLINK_PARAM_ID_CC_CFG_VALID,
	IRDMA_DEVLINK_PARAM_ID_MIN_DEC_FACTOR,
	IRDMA_DEVLINK_PARAM_ID_MIN_RATE,
	IRDMA_DEVLINK_PARAM_ID_DCQCN_F,
	IRDMA_DEVLINK_PARAM_ID_DCQCN_T,
	IRDMA_DEVLINK_PARAM_ID_DCQCN_B,
	IRDMA_DEVLINK_PARAM_ID_RAI_FACTOR,
	IRDMA_DEVLINK_PARAM_ID_HAI_FACTOR,
	IRDMA_DEVLINK_PARAM_ID_RREDUCE_MPERIOD,
	IRDMA_DEVLINK_PARAM_ID_FRAGCNT_LIMIT,
};

#endif /* DEVLINK_SUPPORTED */

/*********************************************************/
#ifdef IRDMA_SET_DRIVER_ID
#define kc_set_driver_id(ibdev) ibdev.driver_id = RDMA_DRIVER_I40IW
#else
#define kc_set_driver_id(x)
#endif /* IRDMA_SET_DRIVER_ID */
/*****************************************************************************/

/********************IRDMA_ALLOC_MW***********************/

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 10, 0)
/**
 * irdma_alloc_mw - Allocate memory window
 * @mw: Memory Window
 * @udata: user data pointer
 */
int irdma_alloc_mw(struct ib_mw *mw, struct ib_udata *udata);
#else
/**
 * irdma_alloc_mw - Allocate memory window
 * @pd: Protection domain
 * @type: Window type
 * @udata: user data pointer
 */
struct ib_mw *irdma_alloc_mw(struct ib_pd *pd, enum ib_mw_type type,
			     struct ib_udata *udata);
#endif /* LINUX_VERSION_CODE */
/*********************************************************/

/******************DEVLINK_RELOAD_DOWN********************/
#ifdef DEVLINK_RELOAD_DOWN_VER_3
int irdma_devlink_reload_down(struct devlink *devlink, bool netns_change,
			      enum devlink_reload_action action,
			      enum devlink_reload_limit limit,
			      struct netlink_ext_ack *extack);
#endif

#ifdef DEVLINK_RELOAD_DOWN_VER_2
int irdma_devlink_reload_down(struct devlink *devlink, bool netns_change,
			      struct netlink_ext_ack *extack);
#endif

struct netlink_ext_ack;
#ifdef DEVLINK_RELOAD_DOWN_VER_1
int irdma_devlink_reload_down(struct devlink *devlink,
			      struct netlink_ext_ack *extack);
#endif /* DEVLINK_RELOAD_DOWN */
/*********************************************************/

#ifdef DEVLINK_RELOAD_UP_VER_3 /*(LINUX_VERSION_CODE >= KERNEL_VERSION(5, 10, 0))*/
int irdma_devlink_reload_up(struct devlink *devlink, enum devlink_reload_action action,
			    enum devlink_reload_limit limit, u32 *actions_performed,
			    struct netlink_ext_ack *extack);
#endif

#ifdef DEVLINK_RELOAD_UP_VER_2 /*defined(RHEL_7_7) || defined(RHEL_7_8) || defined(RHEL_7_9)*/
int irdma_devlink_reload_up(struct devlink *devlink);
#endif

#ifdef DEVLINK_RELOAD_UP_VER_1
int irdma_devlink_reload_up(struct devlink *devlink,
			    struct netlink_ext_ack *extack);
#endif

#ifndef ether_addr_copy
#define ether_addr_copy(mac_addr, new_mac_addr) memcpy(mac_addr, new_mac_addr, ETH_ALEN)
#endif
#ifndef eth_zero_addr
#define eth_zero_addr(mac_addr) memset(mac_addr, 0x00, ETH_ALEN)
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 35)
#define irdma_for_each_ipv6_addr(ifp, tmp, idev) list_for_each_entry_safe(ifp, tmp, &idev->addr_list, if_list)
#else
#define irdma_for_each_ipv6_addr(ifp, tmp, idev) for (ifp = idev->addr_list; ifp != NULL; ifp = ifp->if_next)
#endif /* >= 2.6.35 */

#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 9, 0)
#define netdev_master_upper_dev_get irdma_netdev_master_upper_dev_get
struct net_device *irdma_netdev_master_upper_dev_get(struct net_device *);

#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 1, 0)
#define neigh_release(neigh)
#endif /* < 3.1.0 */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 39)
#define ip_route_output irdma_ip_route_output
struct rtable *irdma_ip_route_output(struct net *, __be32, __be32, u8, int);
#endif /* < 2.6.39 */
#endif /* < 3.9.0 */

#ifdef IB_FW_VERSION_NAME_MAX
void irdma_get_dev_fw_str(struct ib_device *dev, char *str);
#else
void irdma_get_dev_fw_str(struct ib_device *dev, char *str, size_t str_len);
#endif /* IB_FW_VERSION_NAME_MAX */

/*****************************************************************************/

struct dst_entry *irdma_get_fl6_dst(struct sockaddr_in6 *src_addr,
				    struct sockaddr_in6 *dst_addr);
struct neighbour *irdma_get_neigh_ipv6(struct dst_entry *dst,
				       struct sockaddr_in6 *dst_ipaddr);
struct neighbour *irdma_get_neigh_ipv4(struct rtable *rt, __be32 *dst_ipaddr);

#ifdef CREATE_AH_VER_5
int irdma_create_ah_v2(struct ib_ah *ib_ah,
		       struct rdma_ah_attr *attr, u32 flags,
		       struct ib_udata *udata);
int irdma_create_ah(struct ib_ah *ibah,
		    struct rdma_ah_init_attr *attr,
		    struct ib_udata *udata);
#endif

#ifdef CREATE_AH_VER_4
struct ib_ah *irdma_create_ah(struct ib_pd *ibpd,
			      struct rdma_ah_attr *attr,
			      struct ib_udata *udata);
#endif

#ifdef CREATE_AH_VER_3
struct ib_ah *irdma_create_ah(struct ib_pd *ibpd,
			      struct rdma_ah_attr *attr,
			      u32 flags,
			      struct ib_udata *udata);
#endif

#ifdef CREATE_AH_VER_2
int irdma_create_ah(struct ib_ah *ib_ah,
		    struct rdma_ah_attr *attr, u32 flags,
		    struct ib_udata *udata);
#endif

#ifdef CREATE_AH_VER_1_1
struct ib_ah *irdma_create_ah(struct ib_pd *ibpd,
			      struct ib_ah_attr *attr,
			      struct ib_udata *udata);
void irdma_ether_copy(u8 *dmac, struct ib_ah_attr *attr);
#endif
#if defined(CREATE_AH_VER_1_2)
struct ib_ah *irdma_create_ah(struct ib_pd *ibpd,
			      struct rdma_ah_attr *attr,
			      struct ib_udata *udata);
void irdma_ether_copy(u8 *dmac, struct rdma_ah_attr *attr);
#endif

#ifdef DESTROY_AH_VER_4
int irdma_destroy_ah(struct ib_ah *ibah, u32 ah_flags);
#endif

#ifdef DESTROY_AH_VER_3
void irdma_destroy_ah(struct ib_ah *ibah, u32 flags);
#endif

#ifdef DESTROY_AH_VER_2
int irdma_destroy_ah(struct ib_ah *ibah, u32 flags);
#endif

#ifdef DESTROY_AH_VER_1
int irdma_destroy_ah(struct ib_ah *ibah);
#endif

#ifdef IRDMA_DESTROY_CQ_VER_4
int irdma_destroy_cq(struct ib_cq *ib_cq, struct ib_udata *udata);
#endif

#ifdef IRDMA_DESTROY_CQ_VER_3
void irdma_destroy_cq(struct ib_cq *ib_cq, struct ib_udata *udata);
#endif

#ifdef IRDMA_DESTROY_CQ_VER_2
int irdma_destroy_cq(struct ib_cq *ib_cq, struct ib_udata *udata);
#endif

#ifdef IRDMA_DESTROY_CQ_VER_1
int irdma_destroy_cq(struct ib_cq *ib_cq);
#endif

void irdma_unregister_rdma_device(struct ib_device *ibdev);
#ifndef RDMA_MMAP_DB_SUPPORT
int rdma_user_mmap_io(struct ib_ucontext *ucontext, struct vm_area_struct *vma,
		      unsigned long pfn, unsigned long size, pgprot_t prot);
#endif
void irdma_disassociate_ucontext(struct ib_ucontext *context);
int kc_irdma_set_roce_cm_info(struct irdma_qp *iwqp,
			      struct ib_qp_attr *attr,
			      u16 *vlan_id);
int kc_irdma_create_sysfs_file(struct ib_device *ibdev);
struct irdma_device *kc_irdma_get_device(struct net_device *netdev);
void kc_irdma_put_device(struct irdma_device *iwdev);
int irdma_modify_port(struct ib_device *ibdev, u8 port, int mask,
		      struct ib_port_modify *props);

/* Introduced in this series https://lore.kernel.org/linux-rdma/0-v2-270386b7e60b+28f4-umem_1_jgg@nvidia.com/
 * An irdma version helper doing same for older functions with difference that iova is passed in
 * as opposed to derived from umem->iova.
 */
static inline size_t irdma_ib_umem_num_dma_blocks(struct ib_umem *umem, unsigned long pgsz, u64 iova)
{
    /* some older OFED distros do not have ALIGN_DOWN */
#ifndef ALIGN_DOWN
#define ALIGN_DOWN(x, a)	__ALIGN_KERNEL((x) - ((a) - 1), (a))
#endif

	return (size_t)((ALIGN(iova + umem->length, pgsz) -
			 ALIGN_DOWN(iova, pgsz))) / pgsz;
}
#endif /* IRDMA_KCOMPAT_H_ */
