/* SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB */
/* Copyright (c) 2021 Intel Corporation */
/*$FreeBSD$*/
#ifndef FBSD_KCOMPAT_H
#define FBSD_KCOMPAT_H
#include "ice_rdma.h"

#define TASKLET_DATA_TYPE	unsigned long
#define TASKLET_FUNC_TYPE	void (*)(TASKLET_DATA_TYPE)

#define tasklet_setup(tasklet, callback)				\
	tasklet_init((tasklet), (TASKLET_FUNC_TYPE)(callback),		\
		      (TASKLET_DATA_TYPE)(tasklet))

#define from_tasklet(var, callback_tasklet, tasklet_fieldname) \
	container_of(callback_tasklet, typeof(*var), tasklet_fieldname)

#if __FreeBSD_version >= 1202000
#define set_ibdev_dma_device(ibdev, dev) \
	ibdev.dma_device = (dev)
#else
#define set_ibdev_dma_device(ibdev, dev) \
	ibdev.dma_device = device_get_parent(dev)
#endif
#define set_max_sge(props, rf)  \
	((props)->max_sge = (rf)->sc_dev.hw_attrs.uk_attrs.max_hw_wq_frags)
#define kc_set_props_ip_gid_caps(props) \
	((props)->port_cap_flags  |= IB_PORT_IP_BASED_GIDS)
#define rdma_query_gid(ibdev, port, index, gid) \
	ib_get_cached_gid(ibdev, port, index, gid, NULL)
#define kmap(pg) page_address(pg)
#define kunmap(pg)

#define IB_UVERBS_CQ_FLAGS_TIMESTAMP_COMPLETION IB_CQ_FLAGS_TIMESTAMP_COMPLETION

enum ib_port_phys_state {
	IB_PORT_PHYS_STATE_SLEEP = 1,
	IB_PORT_PHYS_STATE_POLLING = 2,
	IB_PORT_PHYS_STATE_DISABLED = 3,
	IB_PORT_PHYS_STATE_PORT_CONFIGURATION_TRAINING = 4,
	IB_PORT_PHYS_STATE_LINK_UP = 5,
	IB_PORT_PHYS_STATE_LINK_ERROR_RECOVERY = 6,
	IB_PORT_PHYS_STATE_PHY_TEST = 7,
};

static inline enum ib_mtu
irdma_mtu_int_to_enum(int mtu)
{
	if (mtu >= 4096)
		return IB_MTU_4096;
	else if (mtu >= 2048)
		return IB_MTU_2048;
	else if (mtu >= 1024)
		return IB_MTU_1024;
	else if (mtu >= 512)
		return IB_MTU_512;
	else
		return IB_MTU_256;
}

struct irdma_tunable_info {
	struct sysctl_ctx_list irdma_sysctl_ctx;
	struct sysctl_oid *irdma_sysctl_tree;
	u8		roce_ena;
};

static inline int
irdma_iw_query_pkey(struct ib_device *ibdev, u8 port, u16 index,
		    u16 *pkey)
{
	*pkey = 0;
	return 0;
}

void		irdma_ether_copy(u8 *dmac, struct ib_ah_attr *attr);

int		irdma_destroy_ah(struct ib_ah *ibah);

enum irdma_status_code
irdma_register_qset(struct irdma_sc_vsi *vsi,
		    struct irdma_ws_node *tc_node);
void
irdma_unregister_qset(struct irdma_sc_vsi *vsi,
		      struct irdma_ws_node *tc_node);
void		ib_unregister_device(struct ib_device *ibdev);
int
rdma_user_mmap_io(struct ib_ucontext *ucontext, struct vm_area_struct *vma,
		  unsigned long pfn, unsigned long size, pgprot_t prot);
void		irdma_disassociate_ucontext(struct ib_ucontext *context);
int
kc_irdma_set_roce_cm_info(struct irdma_qp *iwqp,
			  struct ib_qp_attr *attr,
			  u16 *vlan_id);
struct irdma_device *kc_irdma_get_device(struct net_device *netdev);
void		kc_irdma_put_device(struct irdma_device *iwdev);

void		kc_set_loc_seq_num_mss(struct irdma_cm_node *cm_node);

void		irdma_get_dev_fw_str(struct ib_device *dev, char *str, size_t str_len);

void		irdma_add_ipv6_addr(struct irdma_device *iwdev);
void		irdma_add_ipv4_addr(struct irdma_device *iwdev);
int
irdma_modify_port(struct ib_device *ibdev, u8 port, int mask,
		  struct ib_port_modify *props);
int
irdma_get_dst_mac(struct irdma_cm_node *cm_node, struct sockaddr *dst_sin,
		  u8 *dst_mac);
int
irdma_addr_resolve_neigh(struct irdma_cm_node *cm_node, u32 dst_ip,
			 int arpindex);
int
irdma_addr_resolve_neigh_ipv6(struct irdma_cm_node *cm_node, u32 *dest,
			      int arpindex);
u32		irdma_create_stag(struct irdma_device *iwdev);
void		irdma_free_stag(struct irdma_device *iwdev, u32 stag);
struct irdma_mr;
struct irdma_cq;
struct irdma_cq_buf;
int		irdma_hw_alloc_mw(struct irdma_device *iwdev, struct irdma_mr *iwmr);
struct ib_mw   *
irdma_alloc_mw(struct ib_pd *pd, enum ib_mw_type type,
	       struct ib_udata *udata);
void		irdma_cq_free_rsrc(struct irdma_pci_f *rf, struct irdma_cq *iwcq);
int		irdma_process_resize_list(struct irdma_cq *iwcq, struct irdma_device *iwdev, struct irdma_cq_buf *lcqe_buf);
int		irdma_destroy_cq(struct ib_cq *ib_cq);
struct ib_ucontext *irdma_alloc_ucontext(struct ib_device *, struct ib_udata *);
int		irdma_dealloc_ucontext(struct ib_ucontext *);
struct ib_pd   *irdma_alloc_pd(struct ib_device *, struct ib_ucontext *, struct ib_udata *);
int		irdma_dealloc_pd(struct ib_pd *);
struct ib_ah   *irdma_create_ah(struct ib_pd *, struct ib_ah_attr *, struct ib_udata *);
int		irdma_destroy_ah(struct ib_ah *);
int		irdma_add_gid(struct ib_device *, u8, unsigned int, const union ib_gid *, const struct ib_gid_attr *, void **);
int		irdma_del_gid(struct ib_device *, u8, unsigned int, void **);

/*
 * Introduced in this series https://lore.kernel.org/linux-rdma/0-v2-270386b7e60b+28f4-umem_1_jgg@nvidia.com/ An irdma
 * version helper doing same for older functions with difference that iova is passed in as opposed to derived from
 * umem->iova.
 */
static inline size_t irdma_ib_umem_num_dma_blocks(struct ib_umem *umem, unsigned long pgsz, u64 iova){
	/* some older OFED distros do not have ALIGN_DOWN */
#ifndef ALIGN_DOWN
#define ALIGN_DOWN(x, a)	ALIGN((x) - ((a) - 1), (a))
#endif

	return (size_t) ((ALIGN(iova + umem->length, pgsz) -
			  ALIGN_DOWN(iova, pgsz))) / pgsz;
}

#endif				/* FBSD_KCOMPAT_H */
