/* SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB */
/* Copyright (c) 2019 - 2020 Intel Corporation */
/*$FreeBSD$*/
#include "osdep.h"
#include "type.h"
#include "icrdma_hw.h"

void		configure_flowcontrol(struct irdma_hw *hw);

void		disable_prefetch(struct irdma_hw *hw);

void		disable_tx_spad(struct irdma_hw *hw);

void		rdpu_ackreqpmthresh(struct irdma_hw *hw);

static u32 icrdma_regs[IRDMA_MAX_REGS] = {
	PFPE_CQPTAIL,
	    PFPE_CQPDB,
	    PFPE_CCQPSTATUS,
	    PFPE_CCQPHIGH,
	    PFPE_CCQPLOW,
	    PFPE_CQARM,
	    PFPE_CQACK,
	    PFPE_AEQALLOC,
	    PFPE_CQPERRCODES,
	    PFPE_WQEALLOC,
	    GLINT_DYN_CTL(0),
	    ICRDMA_DB_ADDR_OFFSET,

	    GLPCI_LBARCTRL,
	    GLPE_CPUSTATUS0,
	    GLPE_CPUSTATUS1,
	    GLPE_CPUSTATUS2,
	    PFINT_AEQCTL,
	    GLINT_CEQCTL(0),
	    VSIQF_PE_CTL1(0),
	    PFHMC_PDINV,
	    GLHMC_VFPDINV(0),
	    GLPE_CRITERR,
	    GLINT_RATE(0),
};

static u64 icrdma_masks[IRDMA_MAX_MASKS] = {
	ICRDMA_CCQPSTATUS_CCQP_DONE_M,
	    ICRDMA_CCQPSTATUS_CCQP_ERR_M,
	    ICRDMA_CQPSQ_STAG_PDID_M,
	    ICRDMA_CQPSQ_CQ_CEQID_M,
	    ICRDMA_CQPSQ_CQ_CQID_M,
	    ICRDMA_COMMIT_FPM_CQCNT_M,
};

static u64 icrdma_shifts[IRDMA_MAX_SHIFTS] = {
	ICRDMA_CCQPSTATUS_CCQP_DONE_S,
	    ICRDMA_CCQPSTATUS_CCQP_ERR_S,
	    ICRDMA_CQPSQ_STAG_PDID_S,
	    ICRDMA_CQPSQ_CQ_CEQID_S,
	    ICRDMA_CQPSQ_CQ_CQID_S,
	    ICRDMA_COMMIT_FPM_CQCNT_S,
};

void
icrdma_init_hw(struct irdma_sc_dev *dev)
{
	int		i;
	u8		IOMEM *hw_addr;

	for (i = 0; i < IRDMA_MAX_REGS; ++i) {
		hw_addr = dev->hw->hw_addr;

		if (i == IRDMA_DB_ADDR_OFFSET)
			hw_addr = NULL;

		dev->hw_regs[i] = (u32 IOMEM *) (hw_addr + icrdma_regs[i]);
	}
	dev->hw_attrs.max_hw_vf_fpm_id = IRDMA_MAX_VF_FPM_ID;
	dev->hw_attrs.first_hw_vf_fpm_id = IRDMA_FIRST_VF_FPM_ID;

	for (i = 0; i < IRDMA_MAX_SHIFTS; ++i)
		dev->hw_shifts[i] = icrdma_shifts[i];

	for (i = 0; i < IRDMA_MAX_MASKS; ++i)
		dev->hw_masks[i] = icrdma_masks[i];

	dev->wqe_alloc_db = dev->hw_regs[IRDMA_WQEALLOC];
	dev->cq_arm_db = dev->hw_regs[IRDMA_CQARM];
	dev->aeq_alloc_db = dev->hw_regs[IRDMA_AEQALLOC];
	dev->cqp_db = dev->hw_regs[IRDMA_CQPDB];
	dev->cq_ack_db = dev->hw_regs[IRDMA_CQACK];
	dev->hw_attrs.max_stat_inst = ICRDMA_MAX_STATS_COUNT;

	dev->hw_attrs.uk_attrs.max_hw_wq_size = IRDMA_QP_WQE_MAX_SIZE;
	dev->hw_attrs.uk_attrs.min_sw_wq_size = IRDMA_QP_SW_MIN_WQSIZE;
	dev->hw_attrs.uk_attrs.max_hw_sq_chunk = IRDMA_MAX_QUANTA_PER_WR;
	configure_flowcontrol(dev->hw);
	if (dev->privileged)
		disable_tx_spad(dev->hw);
	if (dev->privileged)
		disable_prefetch(dev->hw);
	if (dev->privileged)
		rdpu_ackreqpmthresh(dev->hw);
	dev->hw_attrs.uk_attrs.feature_flags |= IRDMA_FEATURE_RELAX_RQ_ORDER;
}

#define GLDCB_RTC2PFC_RCB			0x00122100	/* Reset Source: CORER */
#define GLRPB_TC_FRCDROP 0xAC328
#define GLRPB_TC_FRCFC 0xAC32C
#define GLRPB_TC_CFG(_i)			(0x000AC2A4 + ((_i) * 4)) /* _i=0...31 */	/* Reset Source: CORER */
#define GLRPRS_PMCFG_TC_CFG(_i)			(0x00200488 + ((_i) * 4)) /* _i=0...31 */	/* Reset Source: CORER */
#define GLSWT_PMCFG_TC_CFG(_i)			(0x00204900 + ((_i) * 4)) /* _i=0...31 */	/* Reset Source: CORER */
#define GLDCB_TPB_IMM_TPB			0x0009946C	/* Reset Source: CORER */
#define GLDCB_TPB_IMM_TLPM			0x00099468	/* Reset Source: CORER */
#define GLDCB_TLPM_IMM_TCUPM			0x000A018C	/* Reset Source: CORER */
#define GLDCB_TLPM_IMM_TCB			0x000A0190	/* Reset Source: CORER */
#define PRTMAC_HSEC_CTL_RX_ENABLE_GPP		0x001E34C0	/* Reset Source: GLOBR */
#define PRTMAC_HSEC_CTL_RX_ENABLE_GCP		0x001E31C0	/* Reset Source: GLOBR */
#define PRTMAC_HSEC_CTL_RX_PAUSE_ENABLE		0x001E3180	/* Reset Source: GLOBR */

#define PRTMAC_HSEC_CTL_RX_CHECK_MCAST_GCP 0x1E31E0
#define PRTMAC_HSEC_CTL_RX_CHECK_MCAST_GPP 0x1E34E0
#define PRTMAC_HSEC_CTL_RX_CHECK_OPCODE_GPP 0x1E3580
#define PRTMAC_HSEC_CTL_RX_CHECK_OPCODE_GCP 0x1E32E0

#define PRTMAC_HSEC_CTL_TX_PAUSE_ENABLEPRTMAC_HSEC_CTL_TX_PAUSE_ENABLE		0x001E31A0	/* Reset Source: GLOBR */

#define GLRPB_TC2PFC 0xAC040
#define GLDCB_RTC2PFC_RCB			0x00122100	/* Reset Source: CORER */
#define PRTMAC_HSEC_CTL_RX_PAUSE_ENABLE		0x001E3180	/* Reset Source: GLOBR */
#define PRTMAC_HSEC_CTL_TX_PAUSE_ENABLE		0x001E31A0	/* Reset Source: GLOBR */
#define PRTDCB_FCCFG				0x001E4640	/* Reset Source: GLOBR */
#define GLDCB_TC2PFC				0x001D2694	/* Reset Source: CORER */
#define GLPSM_IFCTRL	0xb0790
#define GLDCB_TCUPM_TC2PFC 0xbc34c
#define PRTMAC_HSEC_CTL_TX_PAUSE_REFRESH_TIMER(_i) (0x001E3800 + ((_i) * 32)) /* _i=0...8 */	/* Reset Source: GLOBR */
#define PRTMAC_HSEC_CTL_TX_PAUSE_QUANTA(_i)	(0x001E36E0 + ((_i) * 32)) /* _i=0...8 */	/* Reset Source: GLOBR */
#define PRTMAC_HSEC_CTL_RX_PAUSE_ENABLE		0x001E3180	/* Reset Source: GLOBR */
#define GLDCB_TLPM_TC2PFC 0xa0194

void
configure_flowcontrol(struct irdma_hw *hw)
{
	u32		x;
	u32		port_num = 0;
	u32		number_tc_per_port = 8;
	u64		tc_bit_mask = (number_tc_per_port == 4) ? 0xFUL : 0xFFUL;

	x = rd32(hw, GLDCB_RTC2PFC_RCB);
	x &= ~(tc_bit_mask << (number_tc_per_port * port_num));
	wr32(hw, GLDCB_RTC2PFC_RCB, x);

	x = rd32(hw, GLRPB_TC_FRCDROP);
	x |= (tc_bit_mask << (number_tc_per_port * port_num));
	wr32(hw, GLRPB_TC_FRCDROP, x);

	x = rd32(hw, GLRPB_TC_FRCFC);
	x |= (tc_bit_mask << (number_tc_per_port * port_num));
	wr32(hw, GLRPB_TC_FRCFC, x);

	wr32(hw, GLRPB_TC_CFG(0), 0);
	wr32(hw, GLRPRS_PMCFG_TC_CFG(0), 0);
	wr32(hw, GLSWT_PMCFG_TC_CFG(0), 0);
	wr32(hw, GLDCB_TPB_IMM_TPB, 0xFFFFFFFF);
	wr32(hw, GLDCB_TPB_IMM_TLPM, 0xFFFFFFFF);
	wr32(hw, GLDCB_TLPM_IMM_TCUPM, 0xFFFFFFFF);
	wr32(hw, GLDCB_TLPM_IMM_TCB, 0xFFFFFFFF);

	/* EnableFc */
	wr32(hw, PRTMAC_HSEC_CTL_RX_ENABLE_GPP, 1);

	wr32(hw, PRTMAC_HSEC_CTL_RX_ENABLE_GCP, 1);

	x = rd32(hw, PRTMAC_HSEC_CTL_RX_PAUSE_ENABLE);
	x &= ~0x1FF;
	x |= 0x100;
	wr32(hw, PRTMAC_HSEC_CTL_RX_PAUSE_ENABLE, x);

	wr32(hw, PRTMAC_HSEC_CTL_RX_CHECK_MCAST_GCP, 1);
	wr32(hw, PRTMAC_HSEC_CTL_RX_CHECK_MCAST_GPP, 1);
	wr32(hw, PRTMAC_HSEC_CTL_RX_CHECK_OPCODE_GPP, 1);
	wr32(hw, PRTMAC_HSEC_CTL_RX_CHECK_OPCODE_GCP, 0);
	wr32(hw, PRTMAC_HSEC_CTL_RX_ENABLE_GPP, 1);
	x = rd32(hw, PRTMAC_HSEC_CTL_TX_PAUSE_ENABLE);
	x &= ~0x1FF;
	x |= 0x100;
	wr32(hw, PRTMAC_HSEC_CTL_TX_PAUSE_ENABLE, x);
	wr32(hw, PRTDCB_FCCFG, 1 << 3);

	wr32(hw, PRTMAC_HSEC_CTL_TX_PAUSE_QUANTA(8), 0xFFFF);
	wr32(hw, PRTMAC_HSEC_CTL_TX_PAUSE_REFRESH_TIMER(8), 0x7FFF);

	wr32(hw, GLDCB_TC2PFC, (tc_bit_mask << (number_tc_per_port * port_num)));

	wr32(hw, GLRPB_TC2PFC, (tc_bit_mask << (number_tc_per_port * port_num)));
	wr32(hw, GLDCB_RTC2PFC_RCB, (tc_bit_mask << (number_tc_per_port * port_num)));
	wr32(hw, GLDCB_TLPM_TC2PFC, (tc_bit_mask << (number_tc_per_port * port_num)));
	wr32(hw, GLDCB_TCUPM_TC2PFC, (tc_bit_mask << (number_tc_per_port * port_num)));

	x = rd32(hw, GLPSM_IFCTRL);
	x |= (1 << 16);
	wr32(hw, GLPSM_IFCTRL, x);

	/* stop draining */
	x = rd32(hw, GLRPB_TC_FRCDROP);
	x &= (~(tc_bit_mask << (number_tc_per_port * port_num)));
	wr32(hw, GLRPB_TC_FRCDROP, x);

	x = rd32(hw, GLRPB_TC_FRCFC);
	x &= (~(tc_bit_mask << (number_tc_per_port * port_num)));
	wr32(hw, GLRPB_TC_FRCFC, x);
}

#define GLPE_WQMTXIDXADDR	0x50E000
#define GLPE_WQMTXIDXDATA	0x50E004

void
disable_prefetch(struct irdma_hw *hw)
{
	u32		wqm_data;

	wr32(hw, GLPE_WQMTXIDXADDR, 0x12);
	irdma_mb();

	wqm_data = rd32(hw, GLPE_WQMTXIDXDATA);
	wqm_data &= ~(1);
	wr32(hw, GLPE_WQMTXIDXDATA, wqm_data);
}

void
disable_tx_spad(struct irdma_hw *hw)
{
	u32		wqm_data;

	wr32(hw, GLPE_WQMTXIDXADDR, 0x12);
	irdma_mb();

	wqm_data = rd32(hw, GLPE_WQMTXIDXDATA);
	wqm_data &= ~(1 << 3);
	wr32(hw, GLPE_WQMTXIDXDATA, wqm_data);
}

#define GL_RDPU_CNTRL 		0x52054
void
rdpu_ackreqpmthresh(struct irdma_hw *hw)
{
	u32		val;

	val = rd32(hw, GL_RDPU_CNTRL);
	val &= ~(0x3f << 10);
	val |= (3 << 10);
	wr32(hw, GL_RDPU_CNTRL, val);
}
