# === Configuration loading and preparation ===
$defaultConfigPath = "C:\acl\config\vps-acl-default.json"
$defaultConfig = Get-Content -Raw $defaultConfigPath | ConvertFrom-Json
$extraConfigPath = "C:\acl\config\vps-acl-extra-rules.json"
$extraConfig = Get-Content -Raw $extraConfigPath | ConvertFrom-Json
$logDir = "C:\acl\logs"
$logRetentionDays = 7
$logDate = (Get-Date).ToString("yyyy-MM-dd")
$logPath = Join-Path $logDir "acl-apply-$logDate.log"

if (!(Test-Path -Path $logDir)) {
    New-Item -ItemType Directory -Path $logDir | Out-Null
}

$adminIps = @()
if ($defaultConfig.adminIp -is [string]) { $adminIps += $defaultConfig.adminIp }
elseif ($defaultConfig.adminIp -is [System.Collections.IEnumerable]) { $adminIps += $defaultConfig.adminIp }

$localSubnets = @()
if ($defaultConfig.localSubnet -is [string]) { $localSubnets += $defaultConfig.localSubnet }
elseif ($defaultConfig.localSubnet -is [System.Collections.IEnumerable]) { $localSubnets += $defaultConfig.localSubnet }

$aclBypassVms = @()
if ($extraConfig.PSObject.Properties.Name -contains 'aclBypassVms') { $aclBypassVms = $extraConfig.aclBypassVms }

$ipMap = @{}
foreach ($entry in $extraConfig.vmIpList) { $ipMap[$entry.VMName] = $entry.IPAddress }
$communicationGroups = $extraConfig.communicationGroups

# === Get all running VMs and ACL cache ===
$vms = Get-VM
$vmNames = $vms.Name

# === Clear all ACL rules to ensure a clean environment (optional) ===
foreach ($vmName in $vmNames) {
    $existingAcls = Get-VMNetworkAdapterExtendedAcl -VMName $vmName
    foreach ($acl in $existingAcls) {
        Remove-VMNetworkAdapterExtendedAcl -VMName $vmName -Direction $acl.Direction -Weight $acl.Weight -Confirm:$false
    }
}

# === Initialize weight counters and apply rules ===
foreach ($vmName in $vmNames) {
    if ($aclBypassVms -contains $vmName) {
        $logMessage = "[!] Skip ACL settings (whitelist): $vmName"
        Write-Output $logMessage
        Add-Content -Path $logPath -Value "$((Get-Date).ToString("yyyy-MM-dd HH:mm:ss")) $logMessage"
        continue
    }

    $denyWeight = 1
    $adminWeight = 10

    # 1. Deny local subnets, weights increment from 1
    foreach ($subnet in $localSubnets) {
        Add-VMNetworkAdapterExtendedAcl -VMName $vmName -Action Deny -Direction Inbound -RemoteIPAddress $subnet -Weight $denyWeight
        $logMessage = "[+] Add Deny: $subnet $vmName (weight $denyWeight)"
        Write-Output $logMessage
        Add-Content -Path $logPath -Value "$((Get-Date).ToString("yyyy-MM-dd HH:mm:ss")) $logMessage"
        $denyWeight++
    }

    # 2. Allow administrator IPs, weights increment from 10
    foreach ($adminIp in $adminIps) {
        Add-VMNetworkAdapterExtendedAcl -VMName $vmName -Action Allow -Direction Inbound -RemoteIPAddress $adminIp -Weight $adminWeight
        $logMessage = "[+] Add Allow: Admin IP $adminIp $vmName (weight $adminWeight)"
        Write-Output $logMessage
        Add-Content -Path $logPath -Value "$((Get-Date).ToString("yyyy-MM-dd HH:mm:ss")) $logMessage"
        $adminWeight++
    }
}

# 3. Apply communication group rules
$vmWeightMap = @{}
foreach ($group in $communicationGroups) {
    for ($i=0; $i -lt $group.Count; $i++) {
        for ($j=$i+1; $j -lt $group.Count; $j++) {
            $vm1 = $group[$i]
            $vm2 = $group[$j]

            if (-not ($ipMap.ContainsKey($vm1) -and $ipMap.ContainsKey($vm2))) { continue }
            if (-not ($vmNames -contains $vm1 -and $vmNames -contains $vm2)) { continue }

            $ips1 = $ipMap[$vm1]
            $ips2 = $ipMap[$vm2]

            if (-not $vmWeightMap.ContainsKey($vm1)) { $vmWeightMap[$vm1] = 100 }
            if (-not $vmWeightMap.ContainsKey($vm2)) { $vmWeightMap[$vm2] = 100 }

            foreach ($ip2 in $ips2) {
                Add-VMNetworkAdapterExtendedAcl -VMName $vm1 -Action Allow -Direction Inbound -RemoteIPAddress $ip2 -Weight $vmWeightMap[$vm1]
                $logMessage = "[+] Communication group: $ip2 $vm1 (weight $($vmWeightMap[$vm1]))"
                Write-Output $logMessage
                Add-Content -Path $logPath -Value "$((Get-Date).ToString("yyyy-MM-dd HH:mm:ss")) $logMessage"
                $vmWeightMap[$vm1]++
            }

            foreach ($ip1 in $ips1) {
                Add-VMNetworkAdapterExtendedAcl -VMName $vm2 -Action Allow -Direction Inbound -RemoteIPAddress $ip1 -Weight $vmWeightMap[$vm2]
                $logMessage = "[+] Communication group: $ip1 $vm2 (weight $($vmWeightMap[$vm2]))"
                Write-Output $logMessage
                Add-Content -Path $logPath -Value "$((Get-Date).ToString("yyyy-MM-dd HH:mm:ss")) $logMessage"
                $vmWeightMap[$vm2]++
            }
        }
    }
}

# 4. Apply custom ACLs
$vmCustomWeightMap = @{}
$vmCustomAclMap = @{}
if ($extraConfig.PSObject.Properties.Name -contains 'vmCustomAclList' -and $extraConfig.vmCustomAclList) {
    foreach ($entry in $extraConfig.vmCustomAclList) {
        if ($entry.VMName -and $entry.CustomAcl) {
            $vmCustomAclMap[$entry.VMName] = $entry.CustomAcl
        }
    }
}
foreach ($vm in $vmCustomAclMap.Keys) {
    if (-not $vmCustomWeightMap.ContainsKey($vm)) {
        $vmCustomWeightMap[$vm] = 200
    }

    foreach ($remoteIp in $vmCustomAclMap[$vm]) {
        Add-VMNetworkAdapterExtendedAcl -VMName $vm -Action Allow -Direction Inbound -RemoteIPAddress $remoteIp -Weight $vmCustomWeightMap[$vm]
        $logMessage = "[+] Custom ACL: $remoteIp -> $vm (weight $($vmCustomWeightMap[$vm]))"
        Write-Output $logMessage
        Add-Content -Path $logPath -Value "$((Get-Date).ToString("yyyy-MM-dd HH:mm:ss")) $logMessage"
        $vmCustomWeightMap[$vm]++
    }
}


Get-ChildItem -Path $logDir -Filter "*.log" | Where-Object {
    $_.LastWriteTime -lt (Get-Date).AddDays(-$logRetentionDays)
} | Remove-Item -Force